#pragma warning(disable:4786)

#include <cassert>
#include <sstream>
#if defined(_WIN32)
#include <windows.h>
#else
#include <unistd.h>
#endif

#include "Runtime.h"

ViStatus Runtime::ErrorHandler::status(ViStatus error) {
    if (error < VI_SUCCESS || m_Status == VI_SUCCESS)
        m_Status = error;
    return m_Status;
}

void Runtime::checkRange(const Int16 &value, ViInt16 min, ViInt16 max) {
    if (value.value < min || value.value > max)
        throw value.error;
}

void Runtime::checkRange(const Int32 &value, ViInt32 min, ViInt32 max) {
    if (value.value < min || value.value > max)
        throw value.error;
}

void Runtime::checkRange(const Real64 &value, ViReal64 min, ViReal64 max) {
    if (value.value < min || value.value > max)
        throw value.error;
}

std::string Runtime::dequote(const std::string &s) {
    assert(s.size() > 1);
    return s.substr(1, s.size() - 2);
}

void Runtime::parseList(const std::string &stringList, const bool isQuoted, std::vector<std::string> *vectorList) {
    vectorList->clear();
    std::istringstream in(stringList);
    for (int n = 0; in; n += 1) {
        vectorList->push_back(std::string());
        char c;
        while (in.get(c) && c != ',')
            (*vectorList)[n].append(&c, 1);
    }
    if (isQuoted)
        for (int n = 0; n < vectorList->size(); n += 1)
            (*vectorList)[n] = dequote((*vectorList)[n]);
}

std::string Runtime::quote(const std::string &s) {
    return '"' + s + '"';
}

std::istringstream Runtime::read(const std::string &command) {
    std::string result;
    try {
        m_Session.write(command);
        m_Session.read(result);
    } catch (ViStatus) {
        if (m_IsNotificationEnabled)
            commandComplete();
        throw;
    }
    if (m_IsNotificationEnabled)
        commandComplete();
    return result;
}

void Runtime::readAReal64(const std::string &command, const int size, ViAReal64 array) {
    std::vector<std::string> list;
    readList(command, false, &list);
    for (int n = 0; n < list.size() && n < size; n += 1) {
        std::istringstream in(list[n]);
        in >> array[n];
    }
}

void Runtime::readString(const std::string &command, const bool isQuoted, ViPString result) {
    std::string s;
    readString(command, isQuoted, &s);
    strcpy(result, s.c_str());
}

void Runtime::readString(const std::string &command, const bool isQuoted, std::string *result) {
    *result = isQuoted ? dequote(readString(command)) : readString(command);
}

void Runtime::readAString(const std::string &command, const bool isQuoted, const int size, ViAString array, ViPInt32 numItems) {
    std::vector<std::string> list;
    readList(command, isQuoted, &list);
    for (int n = 0; n < list.size() && n < size; n += 1)
        strcpy(array[n], list[n].c_str());
    *numItems = list.size();
}

void Runtime::readList(const std::string &command, const bool isQuoted, std::vector<std::string> *list) {
    parseList(readString(command), isQuoted, list);
}

std::string Runtime::readString(const std::string &command) {
    std::string result;
    try {
        m_Session.write(command);
        m_Session.read(result);
    } catch (ViStatus) {
        if (m_IsNotificationEnabled)
            commandComplete();
        throw;
    }
    if (m_IsNotificationEnabled)
        commandComplete();
    return result;
}

void Runtime::sleep(ViInt32 milliseconds) {
#ifdef _WIN32
    Sleep(milliseconds);
#else
    ::sleep(milliseconds/1000 + (milliseconds%1000 == 0 ? 0 : 1));
#endif
}

void Runtime::write(const std::string &command) {
    try {
        m_Session.write(command);
    } catch (ViStatus) {
        if (m_IsNotificationEnabled)
            commandComplete();
        throw;
    }
    if (m_IsNotificationEnabled)
        commandComplete();
}

void Runtime::writeBoolean(const std::string &command, ViBoolean value) {
    std::ostringstream out;
    out << command << ' ' << value << std::ends;
    write(out.str());
}

void Runtime::writeABoolean(const std::string &command, ViInt32 size, ViABoolean array) {
    std::ostringstream out;
    out << command << ' ';
    if (size > 0) {
        out << (array[0] == 0 ? 0 : 1);
        for (int n = 1; n < size; n += 1)
            out << ',' << (array[n] == 0 ? 0 : 1);
    }
    write(out.str());
}

void Runtime::writeInt16(const std::string &command, ViInt16 value) {
    std::ostringstream out;
    out << command << ' ' << value << std::ends;
    write(out.str());
}

void Runtime::writeInt32(const std::string &command, ViInt32 value) {
    std::ostringstream out;
    out << command << ' ' << value << std::ends;
    write(out.str());
}

void Runtime::writeReal64(const std::string &command, ViReal64 value) {
    std::ostringstream out;
    out << command << ' ' << value << std::ends;
    write(out.str());
}

void Runtime::writeAReal64(const std::string &command, ViInt32 size, ViAReal64 array) {
    std::ostringstream out;
    out << command << ' ';
    if (size > 0) {
        out << array[0];
        for (int n = 1; n < size; n += 1)
            out << ',' << array[n];
    }
    write(out.str());
}

void Runtime::writeString(const std::string &command, const bool needQuotes, const std::string &value) {
    std::ostringstream out;
    out << command << ' ' << (needQuotes ? quote(value) : value) << std::ends;
    write(out.str());
}

void Runtime::writeAString(const std::string &command, const bool needQuotes, ViInt32 size, ViAString array) {
    std::ostringstream out;;
    out << command << ' ';
    if (size > 0) {
        out << (needQuotes ? quote(array[0]) : std::string(array[0]));
        for (int n = 1; n < size; n += 1)
            out << ',' << (needQuotes ? quote(array[n]) : std::string(array[n]));
    }
    write(out.str());
}
